#import "@preview/touying:0.6.0": *
#import "@preview/codly:1.3.0": *
#import "@preview/cetz:0.3.4"
#import "@preview/fletcher:0.5.7" as fletcher: diagram, node, edge
#import "theme.typ": *
#import "pantoprem.typ": *

#set text(font: "IBM Plex Sans")
#set par(justify: true)
#show: codly-init.with()

#codly(zebra-fill: none, fill: luma(250), stroke: 1pt + luma(220), number-format: none, lang-format: none)

#show raw.where(block: false): it => box(inset: (x: 0.4em), box(outset: 0.4em, radius: 0.1em, fill: luma(240), it))

#show link: it => text(fill: blue, it)

#show: theme.with(
  aspect-ratio: "16-9",
  debug: false,
  precise-size: true,
  guess: false, // true for realtime editing, false for final rendering
  header: self => [],
)

#let smcp(s) = (
  s
    .clusters()
    .map(c => if c.contains(regex("[A-Z]")) {
      c
    } else if c.contains(regex("[a-z]")) {
      text(size: 0.8em, upper(c))
    } else {
      c
    })
    .join()
)

#let msmcp(s) = $upright(#text(size: 0.8em, smcp(s)))$

#let tred(c) = text(fill: red.darken(10%), c)
#let tpurple(c) = text(fill: purple.darken(10%), c)
#let tblue(c) = text(fill: blue.darken(10%), c)
#let tgreen(c) = text(fill: green.darken(10%), c)
#let tgray(c) = text(fill: gray, c)

#let cetz-canvas = touying-reducer.with(reduce: cetz.canvas, cover: cetz.draw.hide.with(bounds: true))

#title-slide(
  footer: [
    Presented by Yanning Chen \@ ProSE Seminar
  ],
)[
  == Navigating Proof Search in Theorem Provers

  === A Brief Tour of ATP in Type Theory
]

== ATP, as a Complexity Problem

#let coord(lo, hi, all, level) = (
  x: 0,
  y: hi - level * (hi - lo) / (all - 1),
  radius: lo + (level - 1) * (hi - lo) / (all - 1),
  offset: 0.8 * (hi - lo) / (all - 1),
)

#let coord_ = coord.with(1.5, 4.1, 4)

#box(width: 100%, height: 80%)[
  #align(center)[
    #cetz-canvas({
      import cetz.draw: *
      circle((coord_(1).x, coord_(1).y), radius: coord_(1).radius, name: "P")
      content("P", text(size: 0.5em)[*P* - CFG])
      (pause,)
      circle((coord_(2).x, coord_(2).y), radius: coord_(2).radius, name: "NP")
      content((rel: (0, coord_(1).offset), to: "NP.south"), text(size: 0.5em)[*NP* - SAT])
      (pause,)
      circle((coord_(3).x, coord_(3).y), radius: coord_(3).radius, name: "PSPACE")
      content((rel: (0, coord_(1).offset), to: "PSPACE.south"), text(size: 0.5em)[*PSPACE* - TQBF])
      (pause,)
      circle((coord_(4).x, coord_(4).y), radius: coord_(4).radius, name: "Elementary")
      content((rel: (0, coord_(1).offset), to: "Elementary.south"), text(size: 0.5em)[*Elementary* - HOL])
      (pause,)
      content((rel: (0, -coord_(1).offset), to: "Elementary.south"), text(size: 0.8em, fill: red)[*RE* - General ATP])
    })
  ]
]

#footnote[https://www.youtube.com/watch?v=VAHyU6gHDdk]

== ATP, as a PL Problem

The good old C.H. Correspondence:\
#box(width: 100%)[
  #align(center)[
    #table(
      columns: (auto, auto),
      inset: 10pt,
      align: horizon + center,
      table.header([*Logic Side*], [*Programming Side*]),
      [proposition], [type],
      [proof], [term],
      [proposition is true], [type has an inhabitant],
      [proposition is false], [type does not have an inhabitant],
      [*proving a proposition*], [*finding an inhabitant*],
    )
  ]
]

#pause

ATP is the *type-inhabitation problem*:\
#align(center)[Given a proposition $P$, find a term of type $P$.]

== How humans prove (in Lean4)

#alternatives[
  ```lean
  inductive Or (a b : Prop) : Prop
  | inl : a → a ∨ b
  | inr : b → a ∨ b
  ```
][
  ```lean
  Or.inl : ∀ {a b : Prop}, a → a ∨ b
  Or.inr : ∀ {a b : Prop}, b → a ∨ b
  Or.elim : ∀ {a b c : Prop}, a ∨ b → (a → c) → (b → c) → c
  ```
]

#pause
#pause

*Theorem*: $forall (p " " q: "Prop"), P or Q -> Q or P$

#alternatives[
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    ?_
  ```
][
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    λ P Q hor => ?_
  ```
][
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    λ P Q hor => Or.elim hor ?_ ?_
  ```
][
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    λ P Q hor => Or.elim hor
      (λ hp => ?_)
      (λ hq => ?_)
  ```
][
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    λ P Q hor => Or.elim hor
      (λ hp => Or.inr ?_)
      (λ hq => Or.inl ?_)
  ```
][
  ```lean
  example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
    λ P Q hor => Or.elim hor
      (λ hp => Or.inr hp)
      (λ hq => Or.inl hq)
  ```
]

== What are these "holes"?

#codly(
  highlights: (
    (line: 3, start: 22, end: 23, fill: purple),
    (line: 4, start: 22, end: 23, fill: purple),
  ),
)
```lean
example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
  λ P Q hor => Or.elim h
    (λ hp => Or.inr ?1)
    (λ hq => Or.inl ?2)
```

#pause

#alternatives[
  - *Metavariable*: A meta-level placeholder for an object-level term. It might have an assigned value (`?1 = hp`), partially assigned (`?x [a ↦ 1]`), or completely unknown.
  - *Goal*: Any metavariable that's not completely assigned.
][
  $
    tpurple(?1): tgray(forall (P " " Q: "Prop")), h_(italic("or")): P or Q, h_p: P tack.r P\
    tpurple(?2): tgray(forall (P " " Q: "Prop")), h_(italic("or")): P or Q, h_q: Q tack.r Q
  $

  *Local Context*: Available premises and the target type for this hole.
]

== How humans prove, alternatively

```lean
example : ∀ (P Q: Prop), P ∨ Q → Q ∨ P := by
  intro p q h
  cases h
  . case inl hp =>
    right
    exact hp
  . case inr hq =>
    left
    exact hq
```

== How humans prove, alternatively

#slide(
  repeat: 7,
  self => [
    #let (uncover, only, alternatives) = utils.methods(self)

    #grid(columns: (1fr, 1.2fr))[
      #block(height: 0.8fr)[
        #text(size: 0.5em)[
          #cetz-canvas({
            import cetz.draw: *
            let self = utils.merge-dicts(
              self,
              config-methods(cover: utils.method-wrapper(hide.with(bounds: true))),
            )
            let (uncover,) = utils.methods(self)

            GoalState(
              { Goal(pos: (0, 0), anchor: "west")[$Entails forall (p, q : Prop), p or q -> q or p$] },
              name: "s1",
            )

            uncover(
              "2-",
              {
                GoalState(
                  { Goal(pos: (1, -1.5), anchor: "west")[$p &: Prop \ Entails &forall (q : Prop), p or q -> q or p$] },
                  name: "s2",
                )
                let start = ("s1.goal.south-west", 10%, "s1.goal.south")
                let bend = (start, "|-", "s2.west")
                EdgeTactic(start, bend, "s2.west")
                LabelTactic(pos: bend, anchor: "base-east")[$#tactic[intro] p$]
              },
            )

            uncover(
              "3-",
              {
                GoalState(
                  { Goal(pos: (2, -3.25), anchor: "west")[$p,q : Prop \ Entails p or q -> q or p$] },
                  name: "s3",
                )
                let start = ("s2.goal.south-west", 10%, "s2.goal.south")
                let bend = (start, "|-", "s3.west")
                EdgeTactic(start, bend, "s3.west")
                LabelTactic(pos: bend, anchor: "base-east")[$#tactic[intro] q$]
              },
            )

            uncover(
              "4-",
              {
                GoalState(
                  { Goal(pos: (3, -5.3), anchor: "west")[$p,q : Prop \ h : p or q \ Entails q or p$] },
                  name: "s4",
                )
                let start = ("s3.goal.south-west", 10%, "s3.goal.south")
                let bend = (start, "|-", "s4.west")
                EdgeTactic(start, bend, "s4.west")
                LabelTactic(pos: bend, anchor: "base-east")[$#tactic[intro] h$]
              },
            )

            uncover(
              "5-",
              {
                GoalState(
                  {
                    Goal(name: "gp", pos: (4, -7.8), anchor: "west")[$p,q : Prop \ h_p : p \ Entails q or p$]
                    Goal(name: "gq", pos: (4, -10), anchor: "west")[$p,q : Prop \ h_q : q \ Entails q or p$]
                  },
                  name: "s5",
                )
                let start = ("s4.goal.south-west", 10%, "s4.goal.south")
                let bend = (start, "|-", "s5.west")
                EdgeTactic(start, bend, "s5.west")
                LabelTactic(pos: bend, anchor: "base-east")[$#tactic[cases] h$]
              },
            )

            uncover(
              "6-",
              {
                GoalState(
                  { Goal(pos: (7.5, -7.8), anchor: "west")[$p,q : Prop \ h_p : p \ Entails p$] },
                  name: "s6p",
                )
                let start = "s5.gp.east"
                EdgeTactic(start, "s6p.west")
                GoalState(
                  { Goal(pos: (7.5, -10), anchor: "west")[$p,q : Prop \ h : q \ Entails q$] },
                  name: "s6q",
                )
                let start = "s5.gq.east"
                EdgeTactic(start, "s6q.west")
              },
            )

            uncover(
              "7-",
              {
                GoalStateEmpty((11, -7.8), name: "s7p")
                let start = "s6p.goal.east"
                EdgeTactic(start, "s7p.west")
                GoalStateEmpty((11, -10), name: "s7q")
                let start = "s6q.goal.east"
                EdgeTactic(start, "s7q.west")
              },
            )
          })
        ]
      ]
    ][
      #set text(size: 0.7em)
      #alternatives[
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          ?_
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P => ?_
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P Q => ?_
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P Q hor => ?_
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P Q hor => Or.elim h
            (λ hp => ?_)
            (λ hq => ?_)
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P Q hor => Or.elim hor
            (λ hp => Or.inr ?_)
            (λ hq => Or.inl ?_)
        ```
      ][
        ```lean
        example: ∀ (P Q: Prop), P ∨ Q → Q ∨ P :=
          λ P Q hor => Or.elim hor
            (λ hp => Or.inr hp)
            (λ hq => Or.inl hq)
        ```
      ]
    ]
  ],
)

== Two Paradigms: term vs tactic

- *Term based*: giving a direct witness of type inhabitation
- *Tactic based*: refining the term by filling "holes" with meta-level instructions

== How humans prove, alternatively

```lean
⟨x: T, p: P x⟩: ∃ x: T, P x
```

#pause

#alternatives[
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := ...
  ```
][
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := by
    refine ⟨3, ?_⟩
    rfl
  ```
][
  #codly(
    highlights: (
      (line: 2, start: 13, end: 13, fill: blue, tag: "-- magical num?"),
    ),
  )
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := by
    refine ⟨3, ?_⟩
    rfl
  ```
][
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := by
    refine ⟨?_, ?_⟩
    pick_goal 2
    rfl
  ```
][
  #codly(
    highlights: (
      (line: 4, start: 3, end: 5, fill: green, tag: "-- closes 1st goal automatically via unification"),
    ),
  )
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := by
    refine ⟨?_, ?_⟩
    pick_goal 2
    rfl
  ```
]
#alternatives(start: 6)[
  *Intuition*:
  + The difficulty of constructing a proof sometimes depends on the order one chooses to fill the holes.
  + Solving a goal might help solve another goal.
][
  *Two Representations*: A view shift from _Search View_ to _Presentation View_.
]

== Three Views of Proofs #footnote[Aniva et al. (TACAS 2025). Pantograph: A Machine-to-Machine Interaction Interface for Advanced Theorem proving, ... in Lean 4.]

- *Presentation View*: Polished proof for presentation (natural language) or verification (machine readable). Contains "_magical_" values.

  #only((2, 3))[
    E.g. "Let $epsilon > 0$. Define $delta := tpurple(min(2epsilon^2, epsilon / 5, 1 / 3))$, ..."
  ]
#pause
#pause
- *Search (proof) View*: The trajectory a prover (machine/human) follows, i.e. _Motivated Proof_#footnote[R. L. Morris. Motivated Proofs: What They Are, Why They Matter and How to Write Them].

  #only(3)[
    "Let $epsilon > 0$. Let $delta := tpurple(?1)$. Using lemma ..., we know $delta <= epsilon / 5$, ... Therefore we can set $delta := min(...)$."
  ]
#only(4)[
  - *Kernel View*: The proof term itself. Might contain metavars (holes) if unfinished.
]

== How humans prove

+ Prove a theory (either by constructing terms by hand or by using tactics) in the *Search View*.
  - Term based: a tree of _terms (with holes) attempted_
  - Tactic based: a tree of _tactic sequence applied_
#pause
+ Close the proof
  - If term based: *Kernel View* proof
  - If tactic based:
    + Polish the proof
    + Get a *Presentation View* proof (→ *Kernel View*)

== How humans prove

#let human-view = diagram(
  spacing: (4em, 0.8em),
  {
    let (search, kernel, presentation, kernel2) = ((0, 1), (1, 0), (1, 2), (2, 2))
    node(search, `Search View`)
    node(kernel, `Kernel View`)
    node(kernel2, `Kernel View`)
    node(presentation, `Presentation View`)
    edge(search, "u", kernel, label: "term", label-side: left, label-pos: 75%, "->")
    edge(search, "d", presentation, label: "tactic", label-pos: 75%, "->")
    edge(search, "d", presentation, label: "polished", label-side: right, label-pos: 75%, "->")
    edge(presentation, kernel2, label: "generate", "-->")
  },
)

#box(width: 100%)
#align(center)[
  #human-view
]

== How machines prove: a historical survey of ATP

+ Brute-force term search (90s)
+ Hammer & SMT bridge (2000-)
+ Data-driven (201?-)

== The brute-force way

The most intuitive and straightforward way.

*Theorem*: $forall (p " " q: "Prop"), P or Q -> Q or P$

#alternatives[
  `?_`
][
  `λ p q h => ?_`
][
  `λ p q h => Or.inl ?_` ⛔
][
  `λ p q h => Or.inr ?_` ⛔
][
  `λ p q h => Or.elim ?_ ?_`
][
  `λ p q z => Or.elim (λ h => ?_) (λ h => ?_)`
][
  `λ p q z => Or.elim (λ h => Or.inr h) (λ h => Or.inl h)`
]

#only("8-")[
  Mimics how humans prove by hand, sans the intuition and experience.
]

#only("9-")[
  #human-view
]

#only("10-")[
  *Problem*: huge search space
]

== The hammer way

#let coord_ = coord.with(1.5, 4.1, 4)

#box(width: 100%)[
  #grid(columns: (1fr, 1fr), gutter: 10pt)[
    #align(center)[
      #cetz-canvas({
        import cetz.draw: *
        circle((coord_(1).x, coord_(1).y), radius: coord_(1).radius, name: "P")
        content("P", text(size: 0.5em)[*P* - CFG])
        circle((coord_(2).x, coord_(2).y), radius: coord_(2).radius, name: "NP")
        content((rel: (0, coord_(1).offset + 0.3), to: "NP.south"), text(size: 0.7em, fill: red)[*NP* - SAT])
        circle((coord_(3).x, coord_(3).y), radius: coord_(3).radius, name: "PSPACE")
        content((rel: (0, coord_(1).offset), to: "PSPACE.south"), text(size: 0.5em, fill: gray)[*PSPACE* - TQBF])
        circle((coord_(4).x, coord_(4).y), radius: coord_(4).radius, name: "Elementary")
        content((rel: (0, coord_(1).offset), to: "Elementary.south"), text(size: 0.5em, fill: gray)[*Elementary* - HOL])
        content(
          (rel: (0, -coord_(1).offset), to: "Elementary.south"),
          text(size: 0.5em, fill: gray)[*RE* - General ATP],
        )
      })
    ]
  ][
    #only("1-")[
      + Cut out a decidable fragment (e.g. QF_uf)
    ]
    #only("2-5")[
      + Find out related theorems
    ]
    #only("6-")[
      + *#tred[Find out related theorems]*
    ]
    #only("3-")[
      + Send them to a solver
    ]
    #only("4-")[
      + Reconstruct the proof
    ]
    #only("5-")[
      #align(center)[
        #diagram(
          spacing: 1.5em,
          label-size: 0.8em,
          {
            let (kernel, external, kernel2) = ((0, 0), (0, 1), (0, 2))
            node(kernel, `Kernel View`)
            node(external, `External Solver`)
            node(kernel2, `Kernel View`)
            edge(kernel, external, label: "embed", label-side: left, "->")
            edge(external, kernel2, label: "reconstruct", label-side: left, "->")
          },
        )
      ]
    ]
  ]
]

#let rainbow(content) = {
  set text(fill: gradient.linear(..color.map.rainbow))
  box(content)
}

== The data-driven way (#strike[#rainbow[The magic blackbox 😨]])

#image("CleanShot 2025-05-14 at 22.03.50@2x.png")

#pause

#align(center)[
  #diagram(
    spacing: 1.5em,
    label-size: 0.7em,
    {
      let (leftmost, presentation, search, presentation2, kernel) = ((-1, 0), (0, 0), (1, 0), (2, 0), (3, 0))
      edge(leftmost, presentation, label: "sketch", "->")
      node(presentation, `Presentation`)
      edge(label: "draft", "->")
      node(search, `Search`)
      edge(label: "prove", "->")
      node(presentation2, `Presentation`)
      edge(label: "generate", "-->")
      node(kernel, `Kernel`)
    },
  )
]

== Challenges

- *Premise Selection*: Which constructor/lemma to use?\
  A huge headache: induction!
#pause
- *Search Space Pruning*: Do not revisit the same state.
#pause
- *Metavar Coupling#footnote[Limperg and Halkjær. (CPP 2023). Aesop: White-Box Best-First Proof Search for Lean.] (Selection)*: Which goal to solve first?

== Metavariable Coupling

Recall
#alternatives[
  #codly(
    highlights: (
      (line: 4, start: 3, end: 5, fill: green, tag: "-- closes 1st goal automatically via unification"),
    ),
  )
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 := by
    refine ⟨?_, ?_⟩
    pick_goal 2
    rfl
  ```
][
  #codly(
    highlights: (
      (line: 2, start: 10, end: 11, fill: purple, tag: "-- " + [fill with `rfl`]),
    ),
  )
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 :=
    ⟨?1, ?2⟩
  ```
][
  #codly(
    highlights: (
      (line: 2, start: 6, end: 7, fill: green, tag: "-- " + [3 by $n=3 ~ 3=3$]),
    ),
  )
  ```lean
  theorem exists_nat_eq_3: ∃ n: Nat, n = 3 :=
    ⟨?1, rfl⟩
  ```
]

#only("4-")[
  There's a dependency between `?1` and `?2`#footnote[Note that while the term "metavar coupling" was first introduced by aesop, the concept itself is not new. It's a natural consequence of dependent unification.].\
  So, the algorithm must be able to ...
]
#only("5-")[+ keep track of the dependencies and update them during unification.]
#only("6-")[+ choose wisely which metavariable to solve first.]

== Case Study

- *Canonical*: Brute-force search, done smartly
- *Hammer*: When type theory based reasoning meets solvers
- *DSP* (*LLM*): The *#rainbow[panacea]*, but at what cost?

== Case Study: Canonical

Conventional term-based ID-DFS, with a special type theory.

#pause

*Core idea*: maintain $beta$-normal $eta$-long form, even through substitution, for free.

#table(
  columns: 2,
  column-gutter: 10pt,
  stroke: none,
  align: horizon,
  $(lambda x. C " " x) " " 1$, [🚫 $beta$-reducible],
  $f: tau_1 -> tau_2$, [🚫 $eta$-expandable],
  $lambda x. f x: tau_1 -> tau_2$, [✅ $beta$-normal $eta$-long],
  $(f g) h$, [🚫 $eta$-expandable: $(lambda x. f g x) h$, i.e. no partial app],
)

== Case Study: Canonical

Maintain $beta$-normal $eta$-long form, even through substitution, for free.

*Why?* It solves the state pruning problem.

All $alpha beta eta$-equiv terms are syntactically equal! ($beta$n$eta$l + ln)\
A hashset of seen terms is enough. No normalization.

== Case Study: Canonical

Maintain $beta$-normal $eta$-long form, even through substitution, for free.

*Why?* It (partly) simplifies the premise selection problem.

E.g. $h " " tpurple(?1)$ with local context $C_1: a -> T, C_2: a -> b -> T, D_1: a -> U tack.r T$

The only two viable candidates are
- $C_1$: $h " " (C_1 " " tpurple(?2))$, and
- $C_2$: $h " " (C_1 " " tpurple(?2) " " tpurple(?3))$

== Case Study: Canonical

Maintain $beta$-normal $eta$-long form, even through substitution, for free.

*Why?* It (partly) solves the premise selection problem.

E.g. $id " " tpurple(?1) " " tgray(?2)$ with local context $C_1: a -> T, C_2: a -> b -> T, D_1: a -> U tack.r ?t$

#alternatives[
  We have three candidates.
][
  If, we are not enforcing $beta$n$eta$l form, it can be a lambda! (We always have $n+1$ choices)

  Worse, $id " " (Pi_(x: a) " " tpurple(?1)) " " tgray(?2) " " tred(?3)$!
][
  But now that we are enforcing $beta$n$eta$l, it's impossible for a metavar to be a lambda:

  $tpurple(?1): tau_1 -> tau_2$ => $lambda x: tau_1. (tpurple(?1): tau_2)$
]

== Case Study: Canonical

Maintain $beta$-normal $eta$-long form, even through substitution, for free.

*Why?* It enforces static arity, enabling usage of efficient data structures.

*Always full app*: `{ head: Term, args: Array Term }`\
*Partial app*: `{ lhs: Term, rhs: { lhs: Term, rhs: ... }}`

== Case Study: Canonical, and its Achilles' Heel

Induction is a big problem.

#pause

#codly(
  highlights: (
    (line: 2, start: 84, end: 89, fill: red),
  ),
)
```lean
Nat.rec : {motive : Nat → Type} →
  motive Nat.zero → ((n : Nat) → motive n → motive n.succ) → (t : Nat) → motive t
```

`motive` is universally quantified. It's a candidate for every hole.

== Case Study: Hammer

Outsourcing the proof search to an external solver (e.g. Z3, Vampire).

#pause
+ Export subgoals to external solvers, often making dependent types opaque
#pause
+ Filter out a list of likely related theorems and send them to the solver
#pause
+ Replaying external proofs back to the type theory via reconstruction

== Case Study: Hammer

Outsourcing the proof search to an external solver (e.g. Z3, Vampire).

#box(width: 100%)[
  #text(size: 0.8em)[
    #align(center)[
      #table(
        columns: 3,
        align: center,
        table.header(
          [],
          [*Hammer*],
          [*ATP*],
        ),

        [*Automation*], [push-button], [varies],
        [*Expressiveness*], [limited to decidable fragment], [full support for DT and HoL],
        [*Trust*], [longer trust chain], [kernel-checked directly],
        [*Performance*], [highly optimized], [slow in large libs or proofs],
        [*Visibility*], [hidden in the solver], [full trace available],
      )
    ]
  ]
]

#pause

Trade-off between expressiveness and automation.

== Case Study: Hammer

Outsourcing the proof search to an external solver (e.g. Z3, Vampire).

Challenges:

+ Premise Selection: depending on other solutions (adhoc heuristics, reinforcement learning, etc.)
#pause
+ Induction: no support (recursors are dependently typed)

== Case Study: DSP (LLM)

The good:

+ *#rainbow[It works magically!]* ($~50%$ success rate on some datasets)
#pause
+ Somewhat explainable: the informal proof sketch is human-readable.

== Case Study: DSP (LLM)

The ugly:

- Limited to tactic based approach due to the nature of DSP, thus
#pause
- Need to tackle with tactic selection, a slightly different problem with premise selection, and
#pause
- Need to learn how to parametrize the tactic.

== Case Study: DSP (LLM)

The bad:

+ Neural networks are not explainable.
#pause
+ Insufficient dataset: most datasets are synthetic, and the real-world datasets are not large enough.
#pause
+ Unreliable benchmarking: data leak problem
  - Avoidable: training set pollution
  - Unavoidable: prior knowledge from natural language pre-training

== Best of all three worlds

+ Symbolic approach with a solver heart\
  Regain the power of domain specific tactics.
#pause
+ Symbolic approach with a neuro heart\
  (Domain-specific) learning based premise/metavar selection.
#pause
+ Neuro approach with a symbolic heart\
  Explainable & reliable logic core.

== Open problems

+ Premise Selection
#pause
+ Lemma Discovery: reuse proofs, generalizing theorems\
  Might help with the induction headache.

== Acknowledgements

_Leni Aniva_, for her brilliant idea of the three views of proof, insights on neuro-symbolic methods, and also her help on the diagrams.

_Tesla Zhang_, for his clear explanation of the Canonical theorem prover.

_Alexander Chichigin_, for his attentive review and constructive criticism.