#import "@preview/touying:0.5.3": *
#import "@preview/codly:1.0.0": *
#import "theme.typ": *

#set text(font: ("IBM Plex Sans"))
#set par(justify: true)
#show: codly-init.with()

#codly(zebra-fill: none, fill: luma(250), stroke: 1pt + luma(220), number-format: none, lang-format: none)

#show raw.where(block: false): it => box(inset: (x: 0.4em), box(outset: 0.4em, radius: 0.1em, fill: luma(240), it))

#show: theme.with(
  aspect-ratio: "16-9",
  // debug: true,
  precise-size: true,
  header: self => [],
)

#let smcp(s) = s.clusters().map(c => if c.contains(regex("[A-Z]")) {
  c
} else if c.contains(regex("[a-z]")) {
  text(size: 0.8em, upper(c))
} else {
  c
}).join()

#let msmcp(s) = $upright(#smcp(s))$

#title-slide(footer: [
  Ralf Jung et al.\@ POPL'18
  #h(1fr)
  Presented by Yanning Chen \@ ProSE Seminar
])[
  = Rustbelt

  == a formalization of Rust type system
]

== Features of Rust type system

- Ownership
- Mutable/Shared References
- Lifetimes
- Interior Mutability

*Goal*: Well-typed Rust programs should be *memory-safe*.

*How?* _Aliasing_ and _mutation_ cannot ocur at the same time on any given location.

== Ownership

In Rust, a type represents:

+ information on what values it can hold
+ *ownership of the resources (e.g. memory)*

*Fact*: Rust uses an affine type system, i.e. a value can be used *at most* once.\
*Consequence*: no two values can own the same resource, so *mutation* allowed but *no* _aliasing_.

```rust
let s1 = String::from("hello");
let s2 = s1; // s1 is moved to s2, i.e. s1 is used and no longer available
println!("{}", s1); // error: use of moved value: s1
```

*Q: What happens when a value is weakened?*\
A: Underlying resources deallocated!

---

== Mutable Reference

What we don't want to gain permanent access to a value?

```rust
fn Vec::push<T>(Vec<T>, T) -> Vec<T>

let v_ = Vec::push(v, 1);   // v is no longer available
```

Instead, Rust uses *mutable references*:

```rust
fn Vec::push<T>(&mut Vec<T>, T)

Vec::push(&mut v, 1);   // v is still available
```

A *mutable reference* grants _temporary exclusive access_ (i.e. _borrowing_) to the value _for the duration of the function call_.\
*Result*: still, _mutation_ allowed but *no* _aliasing_.

---

== Shared Reference

_What if we want to access a value at multiple places?_\
Admit _aliasing_!

```rust
let v = vec![1];
// Two shared references to v are created
join(|| println!("{:?}", &v), || println!("{:?}", &v));
// Still have access to v at the main thread after references are dropped
Vec::push(&mut v, 2);
```

*Result*: for memory safety, allow _aliasing_ but *no* _mutation_.

```rust
let v = vec![1];
let r = &v;             // temporary shared reference
Vec::push(&mut v, 2);   // error: active shared reference exists
println!("{:?}", r);    // shared reference ends here
```

---

== Shared Reference - _Copy_ types

_What if we want to access a value at multiple places?_\
Admit _aliasing_!

Therefore, _shared references_ can be freely duplicated, i.e. _unrestricted variables_ in linear logic.

In Rust, _unrestricted types_ are called _Copy_ types.

Semantically, every type that can be duplicated via bit-wise copy is a _Copy_ type.

- `&T` yes, because it's a shared pointer. `Int` yes, because it's a number.

- `&mut T` no, because it also holds exclusive access. `Vec<Int>` no, because it's a pointer to an heap array, and a bit-wise copy doesn't duplicate the underlying data.

---

== Lifetimes

- *Ownership*: exclusive access, _mutation_
- *Mutable Reference*: _temporary_ exclusive access, _mutation_
- *Shared Reference*: _temporary_ shared access, _aliasing_

How to track if a reference is active? How long is _temporary_?
Answer: equip each reference with a _lifetime_.

```rust
&'a mut T   // mutable reference with lifetime 'a
&'b T       // shared reference with lifetime 'b
```

---

== Lifetimes

#image("index_sig.png", width: 60%)
#image("lifetime_example.png", width: 80%)

- the output of `index_mut` has the same lifetime as the input.
- passing `v` to `index_mut`, we create a lifetime `'b` for `v` and `head`.
- to call `push` we need to create a mutable reference, whose lifetime overlaps with `'b`.

---

== Interior Mutability

*Q*: What if we need shared mutable state? i.e. multi-thread queue?\
*A*: Add primitives that allow _mutation_ through _shared references_, i.e. _interior mutability_.

```rust
Cell::set(&Cell<T>, T)
Cell::get(&Cell<T>) -> T
```

```rust
let c1 : &Cell<i32> = &Cell::new(1);
let c2 : &Cell<i32> = &c1;
c1.set(2);
println!("{}", c2.get()); // 2
```

#pause

*Oops!* _Aliasing_ and _mutation_ at the same time!

`Cell` is implemented using *unsafe* code, i.e. opting _out_ of the type system.

---

== Interior Mutability

If you think about it, `Cell` is still safe to use.

```rust
Cell::set(&Cell<T>, T)
Cell::get(&Cell<T>) -> T
```

`Cell` can only hold _Copy_ types, and returns a copy of the value when `get` is called.

No way to alias the inner data semantically!

---

== Formalization of Rust: Challenges


- Complex language: imperative, traits, ...
- _Unsafe_ types: opting out of syntactic typing rules

---

== Challenge: complex language

*Solution*: work on a subset of Rust intermediate representation called $lambda_("Rust")$.

#image("lrust_skim.png")

---

== Type system of $lambda_("Rust")$

*Observation*: local variables of Rust are also addressable.

*Simplification*: treat local variables as heap-allocated, i.e. _pointer_ types.

- Primitives: `bool`, `int`
- Pointers:
  + $bold("own") tau$: pointer with full ownership of an allocation containing a value of type $tau$
  + $\&_(bold("mut")"/"bold("shr"))^kappa tau$: mutable/shared reference with lifetime $kappa$ to a value of type $tau$
- Other types: $Pi$, $Sigma$, $->$, $mu$, ...

*Note*: Types of local variables of Rust programs are all _pointer_ types.

_Not describing in detail due to time limit._

---

== Challenge: _unsafe_ types

_Unsafe_ types opts out of typing rules, so no way to prove safety from the rules!

#pause

*Solution*: take the _semantic_ approach.

/ *syntactic typing*: terms the typing rules allow to be of type $tau$
/ *semantic typing*: terms that are safe to be treated as type $tau$

---

== Semantic typing

_What is a type?_
#pause
a certain set of values, or, a predicate on values.

*Example*: in lambda calculus with booleans,

- $[|"Bool"|](v) := v = "true" or v = "false"$.
- $[|A * B|](v) := exists v_1, v_2. v = (v_1, v_2) and [|A|](v_1) and [|B|](v_2)$.

---

== Challenges to model Rust type system

- How to describe _ownership_?
- How to describe _temporary_ access?
- How to deal with _interior mutability_?

---

== Challenge: How to describe *ownership*?

_What is a type?_ a certain set of values, or, a predicate on values.

_What predicate? using which logic?_
#pause

*Separation Logic!*

---

== Separation Logic 101

A logic that describes a _heap_.

#let wand = $-#h(-0.3em)*$

- $mono("emp")$: empty heap
- $x |-> v$: heap with a single cell at address $x$ containing value $v$
- $P * Q$: heap that can be _split_ into two parts, one satisfying $P$ and the other satisfying $Q$ (like _conjunction_, but disjoint)
- $P wand Q$: heap that, _disjointly_ combined with another heap satisfying $P$, satisfies $Q$ (like _implication_, but disjoint)

---

== Separation Logic 101

Separation logic is a _substructural_ logic.

*Example*: Consider the following heap: $x = 1$.\
$x |-> 1$ holds, but $x |-> 1 * x |-> 1$ does not. Thus, no _contraction_.

Also, after an implication is applied to a value, the value is _consumed_.

---

== Separation Logic 101

A logic that describes a _heap_.

Separation logic is a _substructural_ logic.

- *Rust types*: a type represents ownership of a resource, and the type system is affine.
- *Separation logic*: a predicate represents a resource, and the logic is _affine_.

Perfect logic to describe Rust types!

_P.S. 🤓👆 Not every separation logic is affine, but the one used in Rustbelt, i.e. Iris, is._

---

== Interpreting Rust types: primitives

Associate every type $tau$ to an _Iris_ (separation logic) predicate on values.

$[|tau|]."own": mono("list Val") -> mono("Prop")$

_(Ignore why we name it "own" for now, will be explained later.)_

- $[|bold("bool")|]."own"(overline(v)) := overline(v) = [bold("true")] or overline(v) = [bold("false")]$
- $[|tau_1 times tau_2|]."own"(
      overline(v)
    ) := exists overline(v_1), overline(v_2). overline(v) = overline(v_1) mono("++") overline(v_2) * [|tau_1|]."own"(
      overline(v_1)
    ) * [|tau_2|]."own"(overline(v_2))$

*Notice*: $*$ is _separating conjunction_, meaning its two oprands are disjoint in memory.

---

== Interpreting Rust types: _Copy_ types

*Recall*: types that can be freely duplicated via bit-wise copy are _Copy_ types.\
*Consequence*: given $[|tau|]."own"(overline(v))$, we can freely duplicate the proposition, recovering contraction rule on the type.

_Proposition that can be freely copied (i.e. $P tack.r P * P$) is called a persistent proposition._

Therefore, the interpretation of _Copy_ types can always be written as:

$[|tau|]."own"(overline(v)) := exists v. overline(v) = [v]. * Phi_tau(v)$, where $Phi_tau$ is a persistent proposition.

E.g. for $tau = bold("bool")$, $Phi_bold("bool")(v) := v = [bold("true")] or v = [bold("false")]$, which is trivially persistent because it's not describing any resource.

---

== Interpreting Rust types: owned pointers

Associate every type $tau$ to an _Iris_ (separation logic) predicate on values.

$[|bold("own") tau|]."own"(overline(v)) := exists cal(l). overline(v) = [
    cal(l)
  ] * exists overline(w). cal(l) |-> overline(w) * [|tau|]."own"(overline(w))$


- $exists cal(l). overline(v) = [cal(l)]$: $overline(v)$ contains a single address $cal(l)$.
- $cal(l) |-> overline(w)$: heap at address $cal(l)$ contains value $overline(w)$.
- $[|tau|]."own"(overline(w))$: $overline(w)$ can be seen as a value of type $tau$.

*Notice*: $*$ is _separating conjunction_, meaning location $cal(l)$ and memory region representing $overline(w)$ are disjoint.

---

== † Interpreting Rust types: owned pointers (for _Copy_ types)

$[|bold("own")" "tau|]."own"(overline(v)) := exists cal(l). overline(v) = [
    cal(l)
  ] * exists overline(w). cal(l) |-> overline(w) * [|tau|]."own"(overline(w))$

*Recall*: types that can be duplicated via bit-wise copy are _Copy_ types.

Try to duplicate $[|bold("own")" "tau|]."own"(overline(v))$:

- $exists cal(l'). overline(v) = [cal(l')]$: can always find another address $cal(l)'$. (assume no allocation failure)
- $exists overline(w'). cal(l') |-> overline(w')$: let $overline(w') = overline(w)$ up to bit-wise copy.
- $[|tau|]."own"(overline(w'))$: holds because $tau$ can be duplicated by bit-wise copy.

*Property*: for any _Copy_ type $tau$, predicate $[|bold("own")" "tau|](overline(v))$ can be freely duplicated.

---

== Interpreting Rust types: mutable references

What's the difference between _mutable references_ and _owned pointers_?

- _owned pointers_: ownership for an unlimited time
- _mutable references_: ownership for _a limited period of time_

---

== Challenge: how to describe _temporary_ ownership?

Recall how we tracked references in Rust type system: _lifetimes_.

*Solution*: lifetime logic.

=== Full borrow predicate

$P$: separation assertion representing ownership of some resource
$\&_(bold("full"))^kappa P$: assertion representing ownership of $P$ _during lifetime $kappa$_

*Intuition*: $P$ holds only when $kappa$ is active.

_We'll head back to the precise definition of lifetime logic later._

---

== Interpreting Rust types: mutable references

$\&_(bold("mut"))^kappa tau$: mutable reference with lifetime $kappa$ to a value of type $tau$

*Meaning*: ownership of a value of type $tau$ for the duration of lifetime $kappa$.

- $[|bold("own") tau|]."own"(overline(v)) := exists cal(l). overline(v) = [
      cal(l)
    ] * exists overline(w). cal(l) |-> overline(w) * [|tau|]."own"(overline(w))$

- $[|\&_bold("mut")^kappa tau|]."own"(overline(v)) := exists cal(l). overline(v) = [cal(l)] * \&_(bold("full"))^(kappa)(
      exists overline(w). cal(l) |-> overline(w) * [|tau|]."own"(overline(w))
    )$

---

== Interpreting Rust types: shared references

$[|\&_(bold("shr"))^kappa tau|]."own"(overline(v)) := ?$

*Question*: what can we say about shared references _universally_?

1. they are pointers
1. they are _Copy_ types, i.e. can be freely duplicated
1. they can be created by downgrading a _mutable reference_
1. for _Copy_ $tau$, we can bit-wise copy the value it points to and get a new $tau$

_Not so interesting!_ Is that true?

*Interior mutability*!

---

== How to deal with _interior mutability_?

Many types have their own sharing reference behavior deviating from the universal rules!

*Solution*: let every type define their own sharing reference behavior, i.e. _sharing predicate_.

- *owned predicate* $[|tau|]."own"(overline(v))$: describe values $overline(v)$ that can be considered as type $tau$
- *sharing predicate* $[|tau|]."shr"(kappa, cal(l))$: describe a location $cal(l)$ and lifetime $kappa$ to be considered as type $\&_(bold("shr"))^kappa tau$

Leveraging the sharing predicate to describe the behavior of shared references.

$[|\&_(bold("shr"))^kappa tau|]."own"(overline(v)) := exists cal(l). overline(v) = [cal(l)] * [|tau|]."shr"(
    kappa, cal(l)
  )$

---

== Interpreting Rust types: shared references

Leveraging the sharing predicate to describe the behavior of shared references.

$[|\&_(bold("shr"))^kappa tau|]."own"(overline(v)) := exists cal(l). overline(v) = [cal(l)] * [|tau|]."shr"(
    kappa, cal(l)
  )$

Laws for sharing predicates:

+ #strike[they are pointers]: already satisfied by the definition of sharing predicate
+ they #strike[are _Copy_ types] can be freely duplicated: $[|tau|]."shr"(kappa, cal(l))$ must be persistent.
+ they can be created by downgrading a _mutable reference_:\ $[|\&_(bold("mut"))^kappa tau|]."own"([cal(l)]) * [kappa]_q wand [|tau|]."shr"(kappa, cal(l)) * [kappa]_q$

_$[kappa]_q$ is a token that asserts the lifetime $kappa$ is active, and we'll talk about it later._

---

== Interpreting Rust types: shared references

4. for _Copy_ $tau$, we can bit-wise copy the value it points to and get a new $tau$.

*Recall*: for _Copy_ types $tau$,\
$[|tau|]."own"(overline(v)) := exists v. overline(v) = [v]. * Phi_(tau)(v)$, where $Phi_(tau)$ is a persistent proposition.

*Define*:\
$[|tau|]."shr"(kappa, cal(l)) := exists v. \&_italic("frac")^(kappa)(cal(l) |->^q v) * Phi_(tau)(v)$

---

== Interpreting Rust types: shared references

*Define*: for _Copy_ types $tau$,\
$[|tau|]."shr"(kappa, cal(l)) := exists v. \&_(italic("frac"))^(kappa)(cal(l) |->^q v) * Phi_(tau)(v)$

*Recall*: for mutable references,\
$[|\&_(bold("mut"))^kappa tau|]."own"(overline(v)) := exists cal(l). overline(v) = [cal(l)] * \&_(bold("full"))^(kappa)(
    exists overline(w). cal(l) |-> overline(w) * [|tau|]."own"(overline(w))
  )$

*Intuition*: † _fractured borrow_ $\&_(italic("frac"))^(kappa) P$ also represents ownership $P$ during lifetime $kappa$, but:

- is _persistent_, because it represents a shared borrow, while full borrow is not
- only grants a fraction of its content ($|->^q$)

_†: no need to understand the details. Just treat them as *full borrow*\s_.

---

== Lifetime logic

Things we used but not defined yet:

- *Full borrow* $\&_(bold("full"))^kappa P$: assertion representing ownership of $P$ _during lifetime $kappa$_
- † *Fractured borrow* $\&_(italic("frac"))^kappa P$: assertion representing ownership of $P$ _during lifetime $kappa$_, but only grants a fraction of its content
- *Lifetime token* $[kappa]_q$: token that asserts the lifetime $kappa$ is active

---

== Lifetime logic

```rust
let mut v = Vec::new();
v.push(0);
{ // <- Vec<i32>
  let mut head = v.index_mut(0);
  *head = 23;
}
println!("{:?}", v);
```

given that

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

---

== Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{
  let mut head = v.index_mut(0); // <- Vec<i32>
```

- need to provide `'a`
- need to pass a mutable reference of lifetime `'a`

---

== Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{
  let mut head = v.index_mut(0); // <- Vec<i32> * [κ] * ([κ] -* [†κ])
```

- need to provide `'a`

  $msmcp("LftL-Begin")$: $mono("True") " "wand" " exists kappa. [kappa]_1 * ([kappa]_1 wand [dagger kappa])$

  Can always

  - create a lifetime token $[kappa]_1$, accompanied by
  - a way to end it $[kappa]_1 wand [dagger kappa]$. ($[dagger kappa]$ is a token that asserts the lifetime $kappa$ has ended.)

---

== Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{
  let mut head = v.index_mut(0); // <- &κ mut Vec<i32> * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

- need to provide `'a` (done)
- need to pass a mutable reference of lifetime `'a`

  $msmcp("LftL-Borrow")$: $P " "wand" " \&_(bold("full"))^(kappa) P * ([dagger kappa] wand P)$

  Given an owned resource $P$, can split it into
  - a full borrow $\&_bold("full")^kappa P$, and
  - an _inheritance_ $[dagger kappa] wand P$ that can retrieve $P$ back after $kappa$ dies.

---

== Lifetime logic: separating conjunction

$msmcp("LftL-Begin")$: $mono("True") " "wand" " exists kappa. [kappa]_1 * ([kappa]_1 wand [dagger kappa])$\
$msmcp("LftL-Borrow")$: $P " "wand" " \&_(bold("full"))^(kappa) P * ([dagger kappa] wand P)$

- *Sep logic* $P * Q$: _heap_ that can be _split_ into two _disjoint_ parts, one satisfying $P$ and the other satisfying $Q$
- *Lifetime logic* $P * Q$: _time_ that can be _split_ into two _disjoint_ parts, one satisfying $P$ (when $kappa$ is alive) and the other satisfying $Q$ (when $kappa$ is dead)

---

== Lifetime logic: frame rule

It's important for $P$ and $Q$ to be _disjoint_.

Consider $P and Q$ and $P * Q$.

$
  (P tack.r P' quad Q tack.r Q') / (P * Q tack.r P' * Q')
  quad "(i.e.)" quad
  (forall x, \{P(x)\} " " c " " \{P'(x)\} quad forall x, \{Q(x)\} " " c " " \{Q'(x)\}) / (forall x, \{(P * Q)(
    x
  )\} " " c " " \{(P' * Q')(x)\})
$

But for $P and Q$,

$
  (forall x, \{P(x)\} " " c " " \{P'(x)\} quad forall x, \{Q(x)\} " " c " " \{Q'(x)\}) / (forall x, \{P(x) and Q(
    x
  )\} " " c " " \{?\})
$

What if $P$ and $Q$ describes some shared resource, and while $P ⊢ P'$, $c$ modifies something that invalidates $Q$?

---

== Lifetime logic: separating conjunction

$msmcp("LftL-Begin")$: $mono("True") " "wand" " exists kappa. [kappa]_1 * ([kappa]_1 wand [dagger kappa])$\
$msmcp("LftL-Borrow")$: $P " "wand" " \&_(bold("full"))^(kappa) P * ([dagger kappa] wand P)$

Whatever we do about $\&_(bold("full"))^(kappa) P$, we can always get back the _inheritance_.

---

== Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{
  let mut head = v.index_mut(0); // <- inside `index_mut`
```

1. split input `&κ Vec<i32>` into the accessed `&κ i32` and the rest `&κ Vec<i32>`
2. return `&k i32` to the caller, and drop the rest

---

== Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{
  let mut head = v.index_mut(0); // <- inside `index_mut`
```

1. split input `&κ Vec<i32>` into the accessed `&κ i32` and the rest `&κ Vec<i32>`
  $msmcp("Lft-bor-split")$: $\&_(bold("full"))^(kappa) (P * Q) tack.r \&_(bold("full"))^(kappa) P * \&_(bold("full"))^(kappa) Q$
2. return `&κ i32` to the caller, and drop the rest
  $P * Q tack.r P$, because _Iris_ is an affine logic

```rust
{
  let mut head = v.index_mut(0); // <- &κ mut i32 * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

---

== Lifetime logic

```rust
  let mut head = v.index_mut(0);
  *head = 23;   // <- i32 * (i32 -* &κ mut i32 * [κ]) * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

Need to access the resource of mutable reference `head`.

$msmcp("Lft-bor-acc")$: $\&_(bold("full"))^(kappa) P * [kappa]_q " "wand" " P * (P wand \&_(bold("full"))^(kappa) P * [kappa]_q)$

Given a full borrow $\&_(bold("full"))^(kappa) P$ and a witness $[kappa]_q$ that shows $kappa$ is active,

- can access the resource $P$, accompanied by
- an _inheritance_ $P wand \&_(bold("full"))^(kappa) P * [kappa]_q$ that can retrieve mutable reference and _lifetime token back_ after the access

_It's important to return things you borrowed!_: _lifetime token_ is such a certificate.

---

== Lifetime logic

```rust
  *head = 23;   // <- &κ mut i32 * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
}
```

```rust
  *head = 23;
}               // <- [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

```rust
  *head = 23;
}               // <- ([†κ] -* Vec<i32>) * [†κ]
```

```rust
  *head = 23;
}               // <- Vec<i32>
```

---

== † Lifetime logic

*Fractured borrow* $\&_(italic("frac"))^(kappa)$ vs *Full borrow* $\&_(bold("full"))^(kappa)$

- *Fractured borrow*\s are persistent: can be accessed simultaneously by multiple parties (freely duplicatable), but do not have full access, i.e. only a fraction of the resource.
- It's always possible to take a little bit of a resource from a *Fractured borrow*, no matter how many times it's been borrowed.

*Intuition*:
- from a full borrow with full lifetime $[kappa]_1$, by downgrading it to a fractured borrow, we can get a fraction of it, thus getting fractional lifetime $[kappa]_q$, e.g. $[kappa]_(0.1)$, which is shorter than $[kappa]_1$.
- The semantics guarantees that we can always get a tiny bit of resource of lifetime $[kappa]_epsilon$ from a fractured borrow.

---

== Proof of soundness

Typing judgments are defined as

$bold(upright(L)) | bold(upright(T)) tack.r upright(I) tack.l x.bold(upright(T))'$

- $bold(upright(L))$ lifetime context
- $bold(upright(T))$ type context
- $upright(I)$ instruction

After the instruction, the type context is updated to $bold(upright(T))'$ with new variable $x$ added.

---

== Proof of soundness

Interpretation of typing judgments:

$bold(upright(L)) | bold(upright(T)) tack.r upright(I) tack.l x.bold(upright(T))' :=
\{[|bold(upright(L))|]_gamma * [|bold(upright(T))|]_gamma\}" "upright(I)" "\{exists v. [|bold(upright(L))|]_gamma * [|bold(upright(T))'|]_(gamma[x <- v])\}$

- Interpreted as a separation logic triple
- $[|bold(upright(T))|]$ uses interpretation of types described earlier

---

== Proof of soundness

+ *FTLR* (Foundamental Theorem of Logical Relations):$forall bold(upright(L)), bold(upright(T)), bold(upright(T')), upright(I). quad bold(upright(L)) | bold(upright(T)) tack.r upright(I) tack.l x.bold(upright(T))' arrow.r.double bold(upright(L)) | bold(upright(T)) tack.double.r upright(I) tack.double.l x.bold(upright(T))'$\
  _Syntactic typing rules are sound w.r.t. semantic typing rules._

+ *Adequacy*: a semantically well-typed program never gets stuck (no invalid memory access or data race).

*Collary*: every rust program that consists of _syntactically_ well-typed _safe_ code and _semantically_ well-typed _unsafe_ code, is safe to execute.

---

== Conclusion

- Rust type system: ownership, mutable/shared references, lifetime, interior mutability
- Formalization: $lambda_"Rust"$, $bold("own")" "tau$, $\&_(bold("mut/shr"))^kappa$. Unsafe types? _Semantic typing_!
- Semantic typing:
  - Separation logic
  - $[| tau |]."own"(overline(v))$, $[| tau |]."shr"(kappa, cal(l))$ (for interior mutability)
  - $\&_(bold("full"))^kappa P$, $\&_(italic("frac"))^kappa P$, $[kappa]_q$? _Lifetime logic_!
- Lifetime logic by example
  - Fractured borrow: persistent + fractional (inclusion) lifetime
- Soundness proof:
  - Judgment interpreted as separation logic triple
  - FTLR (syntactic -> semantic) + Adequacy (semantic -> runtime)

---

== Appendix: Lifetime logic meets Interior Mutability

*Example*: Mutex is a product of flag (true: locked, false: unlocked) and the resource.

$[|bold("mutex")(tau)|]."own"(overline(v)) := [|bold("bool") times tau|]."own"(overline(v))$

$[|bold("mutex")(tau)|]."shr"(κ, cal(l)) := \&_(bold("atom"))^kappa\($\
$quad cal(l) |-> bold("true") or$\
$quad cal(l) |-> bold("false") * \&_(bold("full"))^(kappa)(
    exists overline(v). (cal(l)+1) |-> overline(v) * [|tau|]."own"(overline(v))
  )$\
$\)$

*Atomic persistent borrow* $\&_(bold("atom"))^kappa P$: assertion representing ownership of $P$ that _cannot be accessed for longer than one single instruction cycle_. Can be freely duplicated.

---

== Appendix: Lifetime logic meets Interior Mutability

*Example*: Mutex is a product of flag (true: locked, false: unlocked) and the resource.

$[|bold("mutex")(tau)|]."shr"(κ, cal(l)) := \&_(bold("atom"))^kappa\($\
$quad cal(l) |-> bold("true") or$\
$quad cal(l) |-> bold("false") * \&_(bold("full"))^(kappa)(
    exists overline(v). (cal(l)+1) |-> overline(v) * [|tau|]."own"(overline(v))
  )$\
$\)$

*Atomic persistent borrow* $\&_(bold("atom"))^kappa P$: assertion representing ownership of $P$ that _cannot be accessed for longer than one single instruction cycle_. Can be freely duplicated.

- When unlocked, one thread borrows it, takes its inner full borrow away, and set lock flag. Other threads can't observe an intermediate state due to atomicity.
- Later, another thread tries to borrow it, but the lock flag is set.
- When the first thread releases the lock, it put back the full borrow so another thread can use it.